DROP PROCEDURE [hips].[PatientsWithoutPcehrList]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

-- =============================================
-- Author:        Kirby Bettison
-- Create date:   3 Sep 2013
-- Description:   Gets a list of patients without PCEHR's (Advertised or Disclosed)
-- Modified:
--  3 Feb 2014 - Simon Biber - Make it get patients without PCEHRs
--  6 Feb 2014 - Simon Biber - Avoid duplicate results
-- 13 Feb 2014 - Simon Biber - Show only most recent episode for each patient
-- =============================================
CREATE PROCEDURE [hips].[PatientsWithoutPcehrList] 
(
	@HospitalCodeSystem varchar(20),
	@HospitalId INT = null
)
AS
BEGIN

-- Calculate cutoff to include currently admitted or discharged in the last 24 hours
DECLARE @DischargeDateCutoff DATETIME
SET @DischargeDateCutoff = DATEADD(d, -1, GETDATE())

-- Temporary table to store ID of the most recent matching episode for each
-- patient who matches the conditions in the parameters to this stored procedure
CREATE TABLE #MatchingEpisodes (
	EpisodeId INT NOT NULL
)

-- In case there are multiple episodes with the same admission date, pick the
-- most recently created episode.
INSERT	#MatchingEpisodes
SELECT	MAX(e.EpisodeId)
FROM hips.Episode e
 INNER JOIN (
	-- Determine the admission date of the most recent episode for each patient who
	-- has a current episode in the required hospital and meets the WithPcehr condition.
	SELECT	e.PatientId,
			MAX(e.AdmissionDate) AS MaxAdmit
	FROM	hips.HospitalPatient hp
	 INNER JOIN hips.Episode e
		ON hp.PatientId = e.PatientId
	 LEFT JOIN hips.HealthProviderOrganisationPatient hpop
		ON hp.PatientMasterId = hpop.PatientMasterId
	WHERE (@HospitalId IS NULL OR hp.HospitalId = @HospitalId)
	AND (	(hpop.PcehrAdvertised = 0 AND hpop.PcehrDisclosed = 0)
		OR	(hpop.HealthProviderOrganisationId IS NULL AND hpop.PatientMasterId IS NULL) 
		)
	AND (	e.EpisodeLifecycleId = 11	-- 11 is Admitted, 13 is Discharged
		OR	(e.EpisodeLifecycleId = 13 AND e.DischargeDate > @DischargeDateCutoff)
		)
	GROUP BY e.PatientId
 ) a 
	ON	e.PatientId = a.PatientId
	AND e.AdmissionDate = a.MaxAdmit
WHERE	(	e.EpisodeLifecycleId = 11 
		OR	(e.EpisodeLifecycleId = 13 AND e.DischargeDate > @DischargeDateCutoff)
		)
GROUP BY e.PatientId

SELECT DISTINCT 
        pm.PatientMasterId, 
		pm.CurrentSexId,
		pm.DateOfBirth,
		pm.MedicareNumber,
		pm.MedicareIrn,
		pm.DvaNumber,
		pmn.FamilyName,
		pmn.GivenNames,
		pmn.SuffixId,
		pmn.TitleId,
		pmi.Ihi,
		pmi.IhiStatusId,
		pmi.IhiRecordStatusId,
		pmi.DateLastValidated,
		pmi.RegisteredSexId,
		pmi.RegisteredFamilyName,
		pmi.RegisteredGivenName,
		hp.HospitalId,
		hc.Code AS HospitalCode,
		e.Ward,
		e.Room,
		e.Bed		
FROM #MatchingEpisodes me
 INNER JOIN hips.Episode e
	ON me.EpisodeId = e.EpisodeId
 INNER JOIN hips.HospitalPatient hp
	ON e.PatientId = hp.PatientId
 INNER JOIN hips.PatientMaster pm
	ON hp.PatientMasterId = pm.PatientMasterId
 INNER JOIN hips.PatientMasterName pmn
	ON pm.PatientMasterId = pmn.PatientMasterId
	AND pmn.NameTypeId = 2			-- Current Name Only
 INNER JOIN hips.PatientMasterIhi pmi
	ON pm.PatientMasterId = pmi.PatientMasterId
	AND pmi.IhiStatusId = 1			-- Active IHIs only
	AND pmi.IhiRecordStatusId = 1	-- Verified IHIs only
	AND LEN(pmi.Ihi) = 16			-- Correct length IHIs only
 INNER JOIN hips.HospitalCode hc
	ON hp.HospitalId = hc.HospitalId
	AND hc.CodeSystemId = (SELECT CodeSystemId FROM hips.CodeSystem WHERE Code = @HospitalCodeSystem)

END


GO

